#!/usr/bin/env python3
"""
GUI Demo for Cooperative Command Game

Demonstrates how to use the pygame GUI interface with both standalone
environment and gym wrapper for LLM evaluation and RL training.
"""

import time
import threading
from typing import Optional

try:
    from .gui_interface import GameGUI, GUIConfig
    from .env import CoopCommandEnv, GameConfig, GameDifficulty
    from .gym_wrapper import CoopCommandGymEnv
except ImportError:
    from gui_interface import GameGUI, GUIConfig
    from env import CoopCommandEnv, GameConfig, GameDifficulty
    from gym_wrapper import CoopCommandGymEnv


class GUIWithGymWrapper:
    """
    Combines GUI visualization with gym environment for RL/LLM integration
    """
    
    def __init__(self, difficulty: str = "normal", seed_index: int = 0):
        self.gui_config = GUIConfig()
        self.gui = GameGUI(self.gui_config)
        
        # Create gym environment
        self.gym_env = CoopCommandGymEnv(
            difficulty=difficulty,
            seed_index=seed_index,
            enable_visual=False,
            enable_audio=False
        )
        
        # Initialize
        self.observation, self.info = self.gym_env.reset()
        self.gui.env = self.gym_env._game
        self.gui.current_state = self.gym_env._game.get_game_state()
        self.gui.game_started = True
        
        print(f"GUI + Gym Environment initialized:")
        print(f"  Difficulty: {difficulty}")
        print(f"  Seed: {seed_index}")
        print(f"  Team members: {len(self.gym_env.member_ids)}")
        print(f"  Action space: {self.gym_env.action_space}")

    def step_with_action(self, action):
        """Execute a gym action and update GUI"""
        self.observation, reward, terminated, truncated, self.info = self.gym_env.step(action)
        
        # Update GUI state
        self.gui.current_state = self.gym_env._game.get_game_state()
        
        return self.observation, reward, terminated, truncated, self.info

    def run_with_random_actions(self, max_steps: int = 20):
        """Run the environment with random actions while showing GUI"""
        print(f"\nRunning {max_steps} random actions...")
        
        for step in range(max_steps):
            # Handle GUI events
            self.gui.handle_events()
            if not self.gui.running:
                break
            
            # Take random action
            action = self.gym_env.action_space.sample()
            obs, reward, terminated, truncated, info = self.step_with_action(action)
            
            print(f"Step {step + 1}: Reward={reward:.2f}, Score={info['score_normalized']:.1f}/100")
            
            # Render GUI
            self.gui.render()
            time.sleep(0.5)  # Slow down for visualization
            
            if terminated or truncated:
                print("Game ended!")
                break
        
        print(f"Final score: {self.info['score_normalized']:.1f}/100")

    def run_interactive_mode(self):
        """Run in interactive mode where user can control through GUI"""
        print("\nInteractive mode - use GUI controls:")
        print("  R - Reset environment")
        print("  SPACE - Random action")
        print("  Click map - Move team member")
        print("  ESC - Exit")
        
        while self.gui.running:
            self.gui.handle_events()
            self.gui.render()
            time.sleep(1/60)  # 60 FPS

    def close(self):
        """Cleanup"""
        self.gym_env.close()


def demo_standalone_gui():
    """Demo 1: Standalone GUI with manual control"""
    print("="*60)
    print("DEMO 1: Standalone GUI")
    print("="*60)
    
    gui = GameGUI()
    
    # Automatically create a game
    gui.create_game(difficulty="medium", seed_index=0, enable_audio=False)
    
    print("Controls:")
    print("  R - New game")
    print("  SPACE - Random command")
    print("  Click map - Move team member")
    print("  Mouse wheel - Zoom")
    print("  ESC - Exit")
    
    gui.run()


def demo_gym_integration():
    """Demo 2: GUI integrated with gym environment"""
    print("="*60)
    print("DEMO 2: GUI + Gym Environment Integration")
    print("="*60)
    
    # Create integrated environment
    integrated_env = GUIWithGymWrapper(difficulty="normal", seed_index=1)
    
    try:
        # Run a few random actions
        integrated_env.run_with_random_actions(max_steps=10)
        
        # Switch to interactive mode
        integrated_env.run_interactive_mode()
        
    finally:
        integrated_env.close()


def demo_rl_training_visualization():
    """Demo 3: How this could be used for RL training visualization"""
    print("="*60)
    print("DEMO 3: RL Training Visualization")
    print("="*60)
    
    gui_env = GUIWithGymWrapper(difficulty="hard", seed_index=2)
    
    print("Simulating RL training with GUI visualization...")
    
    # Simulate a simple learning loop
    episode_rewards = []
    
    try:
        for episode in range(3):
            obs, info = gui_env.gym_env.reset()
            gui_env.gui.current_state = gui_env.gym_env._game.get_game_state()
            
            episode_reward = 0
            step_count = 0
            max_steps = 15
            
            print(f"\nEpisode {episode + 1}:")
            
            while step_count < max_steps:
                # Handle GUI events
                gui_env.gui.handle_events()
                if not gui_env.gui.running:
                    break
                
                # Simulate RL agent action (random for demo)
                action = gui_env.gym_env.action_space.sample()
                obs, reward, terminated, truncated, info = gui_env.step_with_action(action)
                
                episode_reward += reward
                step_count += 1
                
                # Render GUI for visualization
                gui_env.gui.render()
                
                print(f"  Step {step_count}: Action={action}, Reward={reward:.2f}, Total={episode_reward:.2f}")
                
                time.sleep(0.3)  # Slow for demo
                
                if terminated or truncated:
                    break
            
            episode_rewards.append(episode_reward)
            print(f"Episode {episode + 1} finished: Total reward = {episode_reward:.2f}")
            time.sleep(1)
        
        print(f"\nTraining summary:")
        print(f"Episode rewards: {[f'{r:.2f}' for r in episode_rewards]}")
        print(f"Average reward: {sum(episode_rewards) / len(episode_rewards):.2f}")
        
        # Keep GUI open for inspection
        print("\nGUI remaining open for inspection. Press ESC to exit.")
        while gui_env.gui.running:
            gui_env.gui.handle_events()
            gui_env.gui.render()
            time.sleep(1/60)
    
    finally:
        gui_env.close()


def demo_llm_evaluation_setup():
    """Demo 4: Setup for LLM evaluation with GUI"""
    print("="*60)
    print("DEMO 4: LLM Evaluation Setup")
    print("="*60)
    
    class LLMEvaluationGUI(GUIWithGymWrapper):
        """Extended GUI for LLM evaluation"""
        
        def __init__(self, *args, **kwargs):
            super().__init__(*args, **kwargs)
            self.command_history = []
            self.llm_responses = []
        
        def execute_llm_command(self, command_text: str, response_text: str = ""):
            """Execute a command from LLM and track it"""
            print(f"LLM Command: {command_text}")
            print(f"LLM Response: {response_text}")
            
            # Parse command and convert to gym action
            # This is a simplified example - real implementation would need
            # proper command parsing from the LLM response
            action = self.gym_env.action_space.sample()  # Placeholder
            
            obs, reward, terminated, truncated, info = self.step_with_action(action)
            
            # Store for analysis
            self.command_history.append(command_text)
            self.llm_responses.append(response_text)
            
            return obs, reward, terminated, truncated, info
        
        def get_evaluation_summary(self):
            """Get summary for LLM evaluation"""
            return {
                'final_score': self.info.get('score_normalized', 0),
                'commands_executed': len(self.command_history),
                'success_rate': self.info.get('success_rate', 0),
                'efficiency': self.info.get('efficiency', 0)
            }
    
    # Create evaluation environment
    eval_env = LLMEvaluationGUI(difficulty="medium", seed_index=0)
    
    try:
        print("Simulating LLM evaluation...")
        
        # Simulate LLM commands
        llm_commands = [
            ("COMMAND: 0 move 30 40", "I'll move the scout to explore the northwest area."),
            ("COMMAND: 1 recon 50 25", "Sending recon to check for hidden objectives."),
            ("COMMAND: 0 status 0 0", "Requesting status update from team.")
        ]
        
        for i, (command, response) in enumerate(llm_commands):
            print(f"\n--- LLM Step {i + 1} ---")
            
            # Execute command with GUI visualization
            obs, reward, terminated, truncated, info = eval_env.execute_llm_command(command, response)
            
            # Update GUI
            eval_env.gui.render()
            
            print(f"Reward: {reward:.2f}, Score: {info['score_normalized']:.1f}/100")
            
            time.sleep(1.5)  # Pause for visualization
            
            if terminated or truncated:
                break
        
        # Show final evaluation
        summary = eval_env.get_evaluation_summary()
        print(f"\nLLM Evaluation Summary:")
        for key, value in summary.items():
            print(f"  {key}: {value}")
        
        print("\nGUI remaining open. Press ESC to exit.")
        while eval_env.gui.running:
            eval_env.gui.handle_events()
            eval_env.gui.render()
            time.sleep(1/60)
    
    finally:
        eval_env.close()


def main():
    """Main demo function"""
    print("🎮 Cooperative Command Game - GUI Demo Suite")
    print("This demonstrates pygame GUI integration with gym environment")
    print()
    
    demos = {
        "1": ("Standalone GUI", demo_standalone_gui),
        "2": ("Gym Integration", demo_gym_integration),
        "3": ("RL Training Visualization", demo_rl_training_visualization),
        "4": ("LLM Evaluation Setup", demo_llm_evaluation_setup)
    }
    
    print("Available demos:")
    for key, (name, _) in demos.items():
        print(f"  {key}. {name}")
    print("  q. Quit")
    
    while True:
        choice = input("\nSelect demo (1-4, q): ").strip().lower()
        
        if choice == 'q':
            print("Goodbye!")
            break
        elif choice in demos:
            name, demo_func = demos[choice]
            print(f"\nStarting: {name}")
            try:
                demo_func()
            except KeyboardInterrupt:
                print("\nDemo interrupted by user.")
            except Exception as e:
                print(f"Demo error: {e}")
                import traceback
                traceback.print_exc()
            print(f"\n{name} completed.")
        else:
            print("Invalid choice. Please select 1-4 or q.")


if __name__ == "__main__":
    main() 